const path = require('path')
const fs = require('fs-extra')

const { EVENTS } = require('./consts')
const Headset = require('./HeadsetSingleton')
const groupsDir = process.env.APPDATA || (process.platform == 'darwin' ? process.env.HOME + '/Library/Preferences' : process.env.HOME + "/.local/share")
const groupsFile = path.join(groupsDir,'../files/groups.json')
const logger = require('../Logger').Logger;

class Groups {

  constructor(ipc) {
    this.ipc = ipc
    this.persistance = true

    //subscribe to events
    this.ipc.on(EVENTS.CREATE_GROUP,(event, arg) => {
      let group = this._generateGroup(arg)
      event.sender.send(`${EVENTS.CREATE_GROUP}-response`,group)
    })


    this.ipc.on(EVENTS.CREATE_GROUP_FOR_BATCHES,(event, arg) => {
      let group = this._generateGroup(arg)
      event.sender.send(`${EVENTS.CREATE_GROUP_FOR_BATCHES}-response`,group)
    })

    this.ipc.on(EVENTS.GET_GROUPS,(event, arg) => {
      let result = this.getAll()
      event.sender.send(`${EVENTS.GET_GROUPS}-response`,result)
    })

    this.ipc.on(EVENTS.DELETE_GROUP,(event, arg) => {
      this.deleteGroup(arg.id)
      event.sender.send(`${EVENTS.DELETE_GROUP}-response`,true)
    })

    this.ipc.on(EVENTS.ADD_DEVICE,(event, arg) => {
      let theInstance = this
      let previouslyCachedDevice = undefined
      if (arg.serial) {
        previouslyCachedDevice = this.getDeviceIdOfSerial(arg.group, arg.serial)
      }

      if (previouslyCachedDevice !== undefined) {
        logger.info('Found cached device when adding to group: ' + previouslyCachedDevice)
        arg.device = previouslyCachedDevice
      }

      var result = Headset.getInstance().setGroup(arg.group, arg.device, function (res, err) {
      if (err) {
          logger.info('Setting a group returned an err.')
          event.sender.send(`${EVENTS.FATAL_USB_ERROR}-push`, {})
          return
          }

        logger.info('Result for setting the group is ', result)
        event.sender.send(`${EVENTS.ADD_DEVICE}-response`,result)
        // Only persist the device in a group if it has no been persisted before, i.e., previouslyCachedDevice is undefined
        if(result && previouslyCachedDevice === undefined) {
          theInstance.addDevice(arg.device, arg.group, arg.serial)
        }
      })
    })

  }

  _generateGroup(arg) {
    if(typeof(arg.name) === 'undefined') {
      let groups = this.getAll()
      arg.name = `group_${groups.length}`
    }
    let group = this.createGroup(arg.name)
    return group
  }

  shouldPersist(persist) {
    this.persistance = persist
  }

  createGroup(name) {

    var groups = this.getAll()
    const reducer = (acc, curr) => acc = curr.id > acc ? curr.id : acc
    let maxGroupId = groups.reduce(reducer,0)

    const group = {
      name: name,
      id: maxGroupId + 1,
      devices: []
    }

    groups.push(group)

    this.saveGroups(groups)

    return group
  }

  deleteGroup(id) {
    if(!this.persistance) return

    var groups = this.getAll()
    groups = groups.filter(group => group.id !== id)
    this.saveGroups(groups)
  }

  saveGroups(groups) {
    if(!this.persistance) return

    if(!fs.existsSync(path.dirname(groupsFile))) {
      fs.mkdirSync(path.dirname(groupsFile))
    }

    fs.writeJSONSync(groupsFile,groups,{flag: 'w', encoding: 'utf8'})
  }


  getAll() {
    var groups = []
    if(!this.persistance) return groups

    if(!fs.existsSync(groupsFile))
    {
      this.saveGroups(groups)
      return groups
    }

    groups = fs.readJSONSync(groupsFile,{encoding: 'utf8'})

    return groups
  }

  getDeviceIdOfSerial(groupId, serial) {
    let groups = this.getAll()
    let group = groups.filter(group => group.id === groupId)
    if (group.length === 0) { return null}
    group = group[0]
    let device = group.devices.filter(function(device) { return device.serial === serial})
    if (device.length === 0) {
      return undefined
    }

    return device[0].id
  }

  addDevice(device, groupId, serial) {
    if(!this.persistance) return

    var groups = this.getAll()
    var group = groups.filter(group => group.id === groupId)
    if(group.length === 0) return null
    group = group[0]
    group.devices.push({id: device, serial:serial})
    this.saveGroups(groups)
  }
}

module.exports = Groups
